@extends("general_base")
@section("title", "My Cart")
@section("style")
<style>
.cart-item {
    background: #fff;
    border-radius: 8px;
    padding: 16px;
    margin-bottom: 12px;
    box-shadow: 0 1px 2px rgba(0,0,0,0.05);
}
.product-header {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-bottom: 8px;
}
.product-image {
    width: 70px;
    height: 70px;
    border-radius: 8px;
    object-fit: cover;
}
.product-info {
    flex-grow: 1;
}
.product-title {
    font-size: 1rem;
    font-weight: 500;
    color: #2c2c2c;
    text-decoration: none;
    display: block;
    margin-bottom: 2px;
}
.product-remove {
    color: #dc3545;
    padding: 8px;
    cursor: pointer;
    opacity: 0.7;
}
.product-remove:hover {
    opacity: 1;
}
.variant-list {
    margin-left: 82px; /* Aligns with product image */
}
.variant-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 8px 0;
    border-bottom: 1px solid #eee;
}
.variant-item:last-child {
    border-bottom: none;
}
.variant-info {
    flex-grow: 1;
}

.variant-controls {
    display: flex;
    align-items: center;
    gap: 12px;
}
.quantity-controls {
    display: flex;
    align-items: stretch;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    height: 32px;
}
.qty-btn {
    width: 32px;
    border: none;
    background: none;
    color: #666;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 0;
}
.qty-btn:hover {
    background: #f8f9fa;
}
.qty-input {
    width: 40px;
    text-align: center;
    border: none;
    font-size: 0.9rem;
    padding: 0;
}
.variant-remove {
    color: #dc3545;
    cursor: pointer;
    opacity: 0.7;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
}
.variant-remove:hover {
    opacity: 1;
}
.cart-summary {
    background: #fff;
    border-radius: 8px;
    padding: 16px;
    position: sticky;
    top: 20px;
    box-shadow: 0 1px 2px rgba(0,0,0,0.05);
}
.summary-title {
    font-size: 1.1rem;
    font-weight: 500;
    color: #2c2c2c;
    margin-bottom: 16px;
}
.summary-breakdown {
    margin-bottom: 12px;
}
.summary-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
}
.summary-row .label {
    font-size: 0.95rem;
    color: #666;
}
.summary-row .value {
    font-size: 0.95rem;
    font-weight: 500;
    color: #2c2c2c;
}
.total-amount {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: 12px;
    margin-top: 12px;
    margin-bottom: 20px;
    border-top: 1px solid #eee;
}
.total-amount .label {
    font-size: 1rem;
    font-weight: 500;
    color: #2c2c2c;
}
.total-amount .value {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--primary-color);
}
.cart-actions {
    display: flex;
    gap: 8px;
    margin-top: 16px;
}
.cart-actions .btn {
    flex: 1;
    padding: 10px;
    border-radius: 6px;
    font-weight: 500;
    text-align: center;
    text-decoration: none;
    font-size: 0.95rem;
}
.continue-shopping {
    background: #f8f9fa;
    color: #2c2c2c;
    border: 1px solid #dee2e6;
}
.continue-shopping:hover {
    background: #f1f3f5;
}
.checkout-btn {
    background: var(--primary-color);
    color: white;
    border: none;
}
.checkout-btn:hover {
    opacity: 0.95;
}
.cart-empty {
    text-align: center;
    padding: 40px 20px;
    background: #fff;
    border-radius: 8px;
    box-shadow: 0 1px 2px rgba(0,0,0,0.05);
}
.cart-empty i {
    font-size: 40px;
    color: #ddd;
    margin-bottom: 12px;
}
.cart-empty p {
    color: #666;
    margin-bottom: 16px;
}
a.continue_shopping_btn {
    color: #313b50;
    display: inline-block;
    text-decoration: underline;
    font-size: 15px;
    line-height: 20px;
    font-weight: 500;
    letter-spacing: 0.8px;
    margin-bottom: 10px;
}
</style>
@endsection
@section("content")
<div class="mn-breadcrumb m-b-30">
    <div class="row">
        <div class="col-12">
            <div class="row gi_breadcrumb_inner">
                <div class="col-md-6 col-sm-12">
                    <h2 class="mn-breadcrumb-title">Shopping Cart</h2>
                </div>
                <div class="col-md-6 col-sm-12">
                    <ul class="mn-breadcrumb-list">
                        <li class="mn-breadcrumb-item"><a href="{{ route('home') }}">Home</a></li>
                        <li class="mn-breadcrumb-item active">My Cart</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<section class="mn-cart-section p-b-15">
    <div class="row">
        <div class="mn-cart-leftside col-lg-8 col-md-12">
            <div id="cartContent">
                <!-- Cart items will be loaded here -->
            </div>
        </div>
        <div class="mn-cart-rightside col-lg-4 col-md-12">
            <div id="cartSummary">
                <!-- Summary will be loaded here -->
            </div>
        </div>
    </div>
</section>
@endsection

@section("script")
<script>
document.addEventListener('DOMContentLoaded', function() {
    let cart = JSON.parse(localStorage.getItem('sb_cart') || '[]');
    loadCart();

    // Listen for cart updates from other components
    window.addEventListener('cartUpdated', function() {
        cart = JSON.parse(localStorage.getItem('sb_cart') || '[]');
        loadCart();
    });

    function loadCart() {
        document.getElementById('cartContent').innerHTML = '<div class="text-center py-4"><div class="spinner-border text-primary" role="status"></div></div>';
        document.getElementById('cartSummary').innerHTML = '<div class="text-center py-4"><div class="spinner-border text-primary" role="status"></div></div>';

        fetch(`${window.location.origin}/cart/items`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(cart),
            credentials: 'same-origin'
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            if (!data || !data.items) {
                throw new Error('Invalid response data');
            }
            renderCart(data.items, data.totals);
        })
        .catch(error => {
            console.error('Error loading cart:', error);
            document.getElementById('cartContent').innerHTML = `
                <div class="cart-empty">
                    <i class="ri-error-warning-line"></i>
                    <p>Error loading cart items. Please try refreshing the page.</p>
                    <a href="{{ route('product.list') }}" class="btn mn-btn-1"><span>Continue Shopping</span></a>
                </div>`;
            document.getElementById('cartSummary').innerHTML = '';
            showNotification('Error loading cart items. Please try refreshing the page.', 'error');
        });
    }

    function renderCart(items, totals) {
        if (items.length === 0) {
            document.getElementById('cartContent').innerHTML = `
                <div class="cart-empty">
                    <i class="ri-shopping-cart-line"></i>
                    <p>Your cart is empty</p>
                    <a href="{{ route('product.list') }}" class="btn mn-btn-1"><span>Continue Shopping</span></a>
                </div>`;
            
            document.getElementById('cartSummary').innerHTML = '';
            return;
        }

        // Group items by product
        const groupedItems = items.reduce((acc, item) => {
            if (!acc[item.id]) {
                acc[item.id] = {
                    id: item.id,
                    title: item.title,
                    image: item.image,
                    url: item.url,
                    variants: []
                };
            }
            acc[item.id].variants.push({
                id: item.variant_id,
                unit: item.unit,
                price: item.price,
                quantity: item.quantity,
                stock: item.stock
            });
            return acc;
        }, {});

        let cartHtml = '';
        Object.values(groupedItems).forEach(product => {
            cartHtml += `
                <div class="cart-item" data-product-id="${product.id}">
                    <div class="product-header">
                        <img src="${product.image}" alt="${product.title}" class="product-image">
                        <div class="product-info">
                            <a href="${product.url}" class="product-title">${product.title}</a>
                        </div>
                        <i class="ri-delete-bin-line product-remove" onclick="removeProduct('${product.id}')"></i>
                    </div>
                    <div class="variant-list">
                        ${product.variants.map(variant => `
                            <div class="variant-item" data-variant-id="${variant.id}">
                                <div class="variant-info">
                                    <div class="variant-name">${variant.unit}</div>
                                    <div class="variant-price">৳${variant.price}</div>
                                    ${variant.stock < 10 ? `<small class="text-danger">(Only ${variant.stock} left in stock)</small>` : ''}
                                </div>
                                <div class="variant-controls">
                                    <div class="quantity-controls">
                                        <button class="qty-btn dec" ${variant.quantity <= 1 ? 'disabled' : ''}>-</button>
                                        <input type="text" class="qty-input" value="${variant.quantity}" min="1" max="${variant.stock}" readonly>
                                        <button class="qty-btn inc" ${variant.quantity >= variant.stock ? 'disabled' : ''}>+</button>
                                    </div>
                                    <i class="ri-close-line variant-remove"></i>
                                </div>
                            </div>
                        `).join('')}
                    </div>
                </div>`;
        });

        document.getElementById('cartContent').innerHTML = cartHtml;

        document.getElementById('cartSummary').innerHTML = `
            <div class="cart-summary">
                <div class="summary-title">Order Summary</div>
                <div class="summary-breakdown">
                    <div class="summary-row">
                        <span class="label">Subtotal</span>
                        <span class="value">৳${totals.sub_total}</span>
                    </div>
                    <div class="summary-row">
                        <span class="label">Delivery Charge</span>
                        <span class="value">৳60</span>
                    </div>
                </div>
                <div class="total-amount">
                    <span class="label">Total</span>
                    <span class="value">৳${totals.sub_total + 60}</span>
                </div>
                <div class="d-flex flex-column align-items-center">
                    <a href="{{ route('product.list') }}" class="continue_shopping_btn"><i class="ri-shopping-cart-fill"></i> Continue Shopping</a>
                    <a href="{{ route('product.checkout') }}" class="mn-btn-2 py-2 px-4"><span>Checkout Now <i class="ri-arrow-right-s-line"></i></span></a>
                </div>
            </div>`;

        setupEventListeners();
    }

    function setupEventListeners() {
        document.querySelectorAll('.qty-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const variantItem = this.closest('.variant-item');
                const productId = this.closest('.cart-item').dataset.productId;
                const variantId = variantItem.dataset.variantId;
                const input = variantItem.querySelector('.qty-input');
                const maxStock = parseInt(input.getAttribute('max'));
                let qty = parseInt(input.value);

                if (this.classList.contains('inc')) {
                    if (qty < maxStock) {
                        qty++;
                    } else {
                        showNotification('Cannot add more. Stock limit reached!', 'error');
                        return;
                    }
                } else if (this.classList.contains('dec') && qty > 1) {
                    qty--;
                }

                updateCartItem(productId, variantId, qty);
            });
        });

        document.querySelectorAll('.variant-remove').forEach(btn => {
            btn.addEventListener('click', function() {
                const variantItem = this.closest('.variant-item');
                const productId = this.closest('.cart-item').dataset.productId;
                const variantId = variantItem.dataset.variantId;
                removeCartItem(productId, variantId);
            });
        });
    }

    function updateCartItem(productId, variantId, quantity) {
        const itemIndex = cart.findIndex(item => 
            item.productId === productId && item.variantId === variantId
        );

        if (itemIndex > -1) {
            cart[itemIndex].quantity = quantity;
            localStorage.setItem('sb_cart', JSON.stringify(cart));
            
            // Dispatch cart update event
            window.dispatchEvent(new CustomEvent('cartUpdated'));
            
            // No need to call loadCart() since the event listener will handle it
            updateCartCount();
        }
    }

    function removeCartItem(productId, variantId) {
        const itemIndex = cart.findIndex(item => 
            item.productId === productId && item.variantId === variantId
        );

        if (itemIndex > -1) {
            cart.splice(itemIndex, 1);
            localStorage.setItem('sb_cart', JSON.stringify(cart));
            
            // Dispatch cart update event
            window.dispatchEvent(new CustomEvent('cartUpdated'));
            
            // No need to call loadCart() since the event listener will handle it
            updateCartCount();
            showNotification('Variant removed from cart', 'success');
        }
    }

    window.removeProduct = function(productId) {
        cart = cart.filter(item => item.productId !== productId);
        localStorage.setItem('sb_cart', JSON.stringify(cart));
        
        // Dispatch cart update event
        window.dispatchEvent(new CustomEvent('cartUpdated'));
        
        // No need to call loadCart() since the event listener will handle it
        updateCartCount();
        showNotification('Product removed from cart', 'success');
    };
});
</script>
@endsection