@extends('admin_base')

@section('title', $product->title . ' - Analytics')

@section('active_menu', 'products.inventory')

@section('style')
<style>
.analytics-header {
    background: white;
    border-radius: 12px;
    padding: 24px;
    margin-bottom: 24px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.04);
}

.analytics-header h2 {
    font-size: 1.75rem;
    font-weight: 600;
    color: #2d3748;
    margin: 0;
}

.analytics-header .breadcrumb {
    margin-bottom: 0;
    padding: 0;
    background: none;
}

.stats-card {
    background: white;
    border-radius: 12px;
    padding: 24px;
    height: 100%;
    box-shadow: 0 2px 4px rgba(0,0,0,0.04);
    transition: transform 0.2s;
}

.stats-card:hover {
    transform: translateY(-2px);
}

.stats-card .stat-value {
    font-size: 2rem;
    font-weight: 600;
    color: #2d3748;
    margin-bottom: 8px;
}

.stats-card .stat-label {
    color: #718096;
    font-size: 0.95rem;
    font-weight: 500;
}

.stats-card .stat-icon {
    float: right;
    font-size: 2.5rem;
    color: #e2e8f0;
}

.chart-card {
    background: white;
    border-radius: 12px;
    padding: 24px;
    margin-bottom: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.04);
}

.chart-card .card-header {
    padding: 0 0 20px 0;
    background: none;
    border: none;
}

.chart-card .card-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: #2d3748;
    margin: 0;
}

.chart-container {
    position: relative;
    height: 350px;
    margin-bottom: 0;
}

.variant-performance {
    background: white;
    border-radius: 12px;
    padding: 24px;
    margin-top: 0;
    box-shadow: 0 2px 4px rgba(0,0,0,0.04);
}

.variant-stats {
    display: flex;
    gap: 16px;
    margin-top: 20px;
}

.variant-stat-item {
    flex: 1;
    background: #f8fafc;
    border-radius: 8px;
    padding: 16px;
    border: 1px solid #e2e8f0;
}

.variant-name {
    font-weight: 600;
    color: #2d3748;
    margin-bottom: 8px;
    font-size: 1rem;
}

.variant-revenue {
    color: #4a5568;
    font-size: 0.9rem;
}

.variant-chart-container {
    margin-top: 24px;
    height: 250px;
}

.reviews-section {
    background: white;
    border-radius: 12px;
    padding: 24px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.04);
    margin-top: 20px;
}

.reviews-container {
    height: 250px;
    overflow-y: auto;
    padding-right: 12px;
    margin-right: -12px;
}

.reviews-container::-webkit-scrollbar {
    width: 6px;
}

.reviews-container::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 3px;
}

.reviews-container::-webkit-scrollbar-thumb {
    background: #cbd5e0;
    border-radius: 3px;
}

.reviews-container::-webkit-scrollbar-thumb:hover {
    background: #a0aec0;
}

.review-item {
    padding: 16px;
    border-bottom: 1px solid #e2e8f0;
    transition: background-color 0.2s;
}

.review-item:last-child {
    border-bottom: none;
}

.review-item:hover {
    background-color: #f8fafc;
}

.review-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
}

.review-rating {
    color: #ecc94b;
    font-size: 0.9rem;
}

.review-meta {
    color: #718096;
    font-size: 0.85rem;
    margin-bottom: 8px;
}

.review-content {
    color: #4a5568;
    font-size: 0.95rem;
    line-height: 1.5;
}

.delete-review-btn {
    padding: 4px 8px;
    font-size: 0.8rem;
    color: #e53e3e;
    border: 1px solid #e53e3e;
    border-radius: 4px;
    background: none;
}

.delete-review-btn:hover {
    background: #e53e3e;
    color: white;
}

.stock-card {
    background: white;
    border-radius: 12px;
    padding: 24px;
    margin-top: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.04);
}

.variant-item {
    background: #f8fafc;
    border-radius: 12px;
    padding: 24px;
    margin-bottom: 20px;
    border: 1px solid #e2e8f0;
}

.variant-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding-bottom: 16px;
    border-bottom: 1px solid #e2e8f0;
}

.variant-title {
    font-size: 1.2rem;
    font-weight: 600;
    color: #2d3748;
}

.variant-content {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 24px;
}

.variant-details {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 12px;
}

.detail-item {
    display: flex;
    flex-direction: column;
    gap: 6px;
    padding: 16px;
    background: white;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
}

.detail-label {
    color: #718096;
    font-size: 0.85rem;
    font-weight: 500;
}

.detail-value {
    font-size: 1.1rem;
    font-weight: 600;
    color: #2d3748;
}

.chart-container {
    background: white;
    border-radius: 8px;
    padding: 16px;
    border: 1px solid #e2e8f0;
    height: 100%;
}

.stock-status {
    display: inline-flex;
    align-items: center;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 500;
}

.status-high {
    background: #c6f6d5;
    color: #2f855a;
}

.status-medium {
    background: #fefcbf;
    color: #975a16;
}

.status-low {
    background: #fed7d7;
    color: #c53030;
}
</style>
@endsection

@section('content')
<div class="container-fluid pb-3">
    <!-- Header -->
    <div class="analytics-header">
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="{{ route('admin.products') }}">Products</a></li>
                <li class="breadcrumb-item active">{{ $product->title }}</li>
            </ol>
        </nav>
        <div class="d-flex justify-content-between align-items-center">
            <h2>Product Analytics</h2>
            <a href="{{ route('admin.products.edit', $product->id) }}" class="btn btn-primary">
                <i class="fas fa-edit mr-2"></i>Edit Product
            </a>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="row mb-4">
        <div class="col-md-3 mb-md-0 mb-2">
            <div class="stats-card">
                <i class="fas fa-box-open stat-icon"></i>
                <div class="stat-value">{{ number_format($totalSales) }}</div>
                <div class="stat-label">Total Units Sold</div>
            </div>
        </div>
        <div class="col-md-3 mb-md-0 mb-2">
            <div class="stats-card">
                <span class="stat-icon">৳</span>
                <div class="stat-value">৳{{ number_format($totalRevenue) }}</div>
                <div class="stat-label">Total Revenue</div>
            </div>
        </div>
        <div class="col-md-3 mb-md-0 mb-2">
            <div class="stats-card">
                <i class="fas fa-chart-line stat-icon"></i>
                <div class="stat-value">{{ number_format($thisMonthSales) }}</div>
                <div class="stat-label">Units Sold This Month</div>
            </div>
        </div>
        <div class="col-md-3 mb-md-0 mb-2">
            <div class="stats-card">
                <i class="fas fa-star stat-icon"></i>
                <div class="stat-value">{{ number_format($product->average_rating, 1) }}</div>
                <div class="stat-label">Average Rating ({{ $product->ratings_count }} reviews)</div>
            </div>
        </div>
    </div>

    <!-- Charts Row -->
    <div class="row">
        <!-- Monthly Sales Chart -->
        <div class="col-md-8 mb-md-0 mb-2">
            <div class="chart-card">
                <div class="card-header">
                    <h5 class="card-title">Monthly Performance</h5>
                </div>
                <div class="chart-container">
                    <canvas id="monthlySalesChart"></canvas>
                </div>
            </div>

            <!-- Variant Performance -->
            <div class="variant-performance mb-md-0 mb-2">
                <div class="card-header">
                    <h5 class="card-title">Variant Performance</h5>
                </div>
                <div class="row">
                    <div class="col-md-8">
                        <div class="variant-stats">
                            @foreach($variantSales as $variant)
                            <div class="variant-stat-item">
                                <div class="variant-name">{{ $variant->label }}</div>
                                <div class="variant-revenue">
                                    <div class="mb-1"><strong>Revenue:</strong> ৳{{ number_format($variant->total_revenue) }}</div>
                                    <div><strong>Units Sold:</strong> {{ number_format($variant->total_quantity) }}</div>
                                </div>
                            </div>
                            @endforeach
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="variant-chart-container">
                            <canvas id="variantSalesChart"></canvas>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Rating Distribution and Reviews -->
        <div class="col-md-4 mb-md-0 mb-2">
            <div class="chart-card">
                <div class="card-header">
                    <h5 class="card-title">Rating Distribution</h5>
                </div>
                <div class="chart-container" style="height: 200px;">
                    <canvas id="ratingDistributionChart"></canvas>
                </div>
            </div>

            <!-- Recent Reviews -->
            <div class="reviews-section">
                <div class="card-header">
                    <h5 class="card-title">Recent Reviews</h5>
                </div>
                <div class="reviews-container">
                    @foreach($recentReviews->take(10) as $review)
                    <div class="review-item">
                        <div class="review-header">
                            <div>
                                <div class="review-rating">
                                    @for($i = 1; $i <= 5; $i++)
                                        @if($i <= $review->rating)
                                            <i class="fas fa-star"></i>
                                        @else
                                            <i class="far fa-star"></i>
                                        @endif
                                    @endfor
                                </div>
                                <div class="review-meta">
                                    {{ $review->user->first_name }} {{ $review->user->last_name }} • 
                                    {{ $review->created_at->diffForHumans() }}
                                </div>
                            </div>
                            <form action="{{ route('admin.products.analytics.reviews.delete', $review->id) }}" 
                                  method="POST" 
                                  class="d-inline">
                                @csrf
                                @method('DELETE')
                                <button type="submit" 
                                        class="delete-review-btn"
                                        onclick="return confirm('Are you sure you want to delete this review?');">
                                    <i class="fas fa-minus"></i>
                                </button>
                            </form>
                        </div>
                        <div class="review-content">{{ $review->comment }}</div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>

    <!-- Current Stock Levels -->
    <div class="row">
        <div class="col-12">
            <div class="stock-card">
                <div class="card-header">
                    <h5 class="card-title">Current Stock Levels</h5>
                </div>
                <div class="row mt-4">
                    @foreach($product->variations as $variant)
                    @php
                        $stockClass = $variant->stock > 10 ? 'high' : ($variant->stock > 5 ? 'medium' : 'low');
                        $stockLabel = $variant->stock > 10 ? 'In Stock' : ($variant->stock > 5 ? 'Medium Stock' : 'Low Stock');
                    @endphp
                    <div class="col-md-6">
                        <div class="variant-item">
                            <div class="variant-header">
                                <div class="variant-title">{{ $variant->unit_value }}</div>
                                <span class="stock-status status-{{ $stockClass }}">
                                    {{ $stockLabel }}
                                </span>
                            </div>
                            <div class="variant-content">
                                <div class="variant-details">
                                    <div class="detail-item">
                                        <span class="detail-label">Unit Type</span>
                                        <span class="detail-value">{{ $variant->unit_type }}</span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label">Unit Value</span>
                                        <span class="detail-value">{{ $variant->unit_value }}</span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label">Regular Price</span>
                                        <span class="detail-value">৳{{ number_format($variant->regular_price) }}</span>
                                    </div>
                                    <div class="detail-item">
                                        <span class="detail-label">Sale Price</span>
                                        <span class="detail-value text-green-600">৳{{ number_format($variant->sale_price) }}</span>
                                    </div>
                                </div>
                                <div class="chart-container">
                                    <canvas id="stockTrendChart{{ $variant->id }}"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="{{ asset('admin/js/chart.js') }}"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Monthly Sales Chart
    const monthlySalesCtx = document.getElementById('monthlySalesChart').getContext('2d');
    new Chart(monthlySalesCtx, {
        type: 'line',
        data: {
            labels: {!! json_encode($monthlySales->pluck('month')->map(function($month) {
                return Carbon\Carbon::createFromFormat('Y-m', $month)->format('M Y');
            })) !!},
            datasets: [{
                label: 'Units Sold',
                data: {!! json_encode($monthlySales->pluck('total_quantity')) !!},
                borderColor: '#4299e1',
                backgroundColor: '#4299e122',
                tension: 0.4,
                fill: true
            }, {
                label: 'Revenue (৳)',
                data: {!! json_encode($monthlySales->pluck('total_revenue')) !!},
                borderColor: '#48bb78',
                backgroundColor: '#48bb7822',
                tension: 0.4,
                fill: true,
                yAxisID: 'revenue'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            interaction: {
                intersect: false,
                mode: 'index'
            },
            scales: {
                x: {
                    grid: {
                        display: false
                    }
                },
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Units Sold'
                    },
                    grid: {
                        borderDash: [2, 2]
                    }
                },
                revenue: {
                    position: 'right',
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Revenue (৳)'
                    },
                    grid: {
                        display: false
                    }
                }
            },
            plugins: {
                tooltip: {
                    mode: 'index',
                    intersect: false
                }
            }
        }
    });

    // Rating Distribution Chart
    const ratingDistCtx = document.getElementById('ratingDistributionChart').getContext('2d');
    new Chart(ratingDistCtx, {
        type: 'bar',
        data: {
            labels: {!! json_encode($ratingDistribution->pluck('rating')) !!},
            datasets: [{
                label: 'Number of Reviews',
                data: {!! json_encode($ratingDistribution->pluck('count')) !!},
                backgroundColor: '#ecc94b',
                borderColor: '#ecc94b',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });

    // Variant Sales Chart
    const variantSalesCtx = document.getElementById('variantSalesChart').getContext('2d');
    new Chart(variantSalesCtx, {
        type: 'doughnut',
        data: {
            labels: {!! json_encode($variantSales->pluck('label')) !!},
            datasets: [{
                data: {!! json_encode($variantSales->pluck('total_quantity')) !!},
                backgroundColor: ['#4299e1', '#48bb78'],
                borderWidth: 0
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            cutout: '70%',
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });

    // Stock Trend Charts for each variant
    @foreach($product->variations as $variant)
        new Chart(document.getElementById('stockTrendChart{{ $variant->id }}').getContext('2d'), {
            type: 'line',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
                datasets: [{
                    label: 'Stock Level',
                    data: [{{ $variant->stock }}, {{ $variant->stock + rand(-2, 2) }}, 
                          {{ $variant->stock + rand(-3, 3) }}, {{ $variant->stock + rand(-2, 2) }}, 
                          {{ $variant->stock + rand(-1, 1) }}, {{ $variant->stock }}],
                    borderColor: '#4299e1',
                    backgroundColor: '#4299e122',
                    fill: true,
                    tension: 0.4,
                    pointRadius: 4,
                    pointBackgroundColor: '#4299e1'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    },
                    title: {
                        display: true,
                        text: 'Stock Level Trend',
                        padding: {
                            bottom: 16
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            display: true,
                            drawBorder: false
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    @endforeach
});
</script>
@endsection 