<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Pages;
use App\Models\Faq;
use App\Mail\ContactFormSubmission;
use Illuminate\Support\Facades\Mail;
use App\Models\Category;
use App\Models\Product;
use App\Models\BestDeals;
use App\Models\Customization;

class GeneralController extends Controller
{
    public function home()
    {
        // Get categories with their products for the category section
        $categories = Category::with(['products.images'])
            ->whereHas('products', function($query) {
                $query->selectRaw('category_id, COUNT(*) as product_count')
                    ->groupBy('category_id')
                    ->havingRaw('COUNT(*) >= 3');
            })
            ->get()
            ->map(function($category) {
                return [
                    'name' => $category->name,
                    'slug' => $category->slug,
                    'product_count' => $category->products->count(),
                    'random_products' => $category->products()
                        ->with('images')
                        ->inRandomOrder()
                        ->take(3)
                        ->get()
                ];
            });

        // Get latest products
        $latestProducts = Product::with(['brand','variations' => function($q) {
            $q->where('default', true);
        }, 'images' => function($q) {
            $q->where('is_primary', true);
        }, 'category'])
        ->withCount([
            'variations as variants_count' => function($q) {},
            'variations as in_stock_variants_count' => function($q) {
                $q->where('in_stock', '>', 0);
            }
        ])
        ->latest()
        ->take(12)
        ->get();

        // Get best deal products
        $bestDealProducts = collect([]);
        $activeDeal = BestDeals::active()->latest()->first();
        if ($activeDeal) {
            $bestDealProducts = $activeDeal->products()
                ->with(['brand','variations' => function($q) {
                    $q->where('default', true);
                }, 'images' => function($q) {
                    $q->where('is_primary', true);
                }, 'category'])
                ->withCount([
                    'variations as variants_count' => function($q) {},
                    'variations as in_stock_variants_count' => function($q) {
                        $q->where('in_stock', '>', 0);
                    }
                ])
                ->take(8)
                ->get();
        }

        // Load customization
        $homeCustomization = Customization::singleton()->home ?? [
            'sliders' => [],
            'services' => [],
            'featured' => [],
            'hero_boxes' => [],
        ];

        $data = [
            "categories" => $categories,
            "latestProducts" => $latestProducts,
            "bestDealProducts" => $bestDealProducts,
            "dealEndDate" => $activeDeal ? $activeDeal->end_date : null,
            "activeDeal" => $activeDeal,
            "homeCustomization" => $homeCustomization,
        ];
        
        return view('general.home', $data);
    }

    public function bestDeals()
    {
        $activeDeal = BestDeals::with(['products' => function($query) {
            $query->with([
                'variations' => function($q) {
                    $q->orderBy('regular_price', 'asc')
                      ->where('in_stock', '>', 0);
                },
                'images' => function($q) {
                    $q->where('is_primary', true);
                },
                'category'
            ])
            ->withCount([
                'variations as variants_count' => function($q) {},
                'variations as in_stock_variants_count' => function($q) {
                    $q->where('in_stock', '>', 0);
                }
            ]);
        }])->active()->latest()->first();

        $products = $activeDeal ? $activeDeal->products()->paginate(24)->withQueryString() : collect([]);

        return view('general.best_deals', [
            'products' => $products,
            'dealName' => $activeDeal ? $activeDeal->name : null,
            'dealDescription' => $activeDeal ? $activeDeal->description : null,
            'dealEndDate' => $activeDeal ? $activeDeal->end_date : null
        ]);
    }
    
    

    public function page($slug)
    {
        $page = Pages::where('slug', $slug)->firstOrFail();
        return view('general.page', compact('page'));
    }

    public function faq()
    {
        $faqs = Faq::all();
        return view('general.faq', compact('faqs'));
    }

    

    public function contact()
    {
        $customization = Customization::singleton();
        $contact = $customization->contact ?? [
            'address' => null,
            'phone' => null,
            'email' => null,
            'map_embed' => null
        ];

        return view('general.contact', compact('contact'));
    }

    public function submitContact(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'type' => 'required|in:general,order,return,account,feedback,complaint',
            'message' => 'required|string',
            'g-recaptcha-response' => 'required|captcha'
        ], [
            'g-recaptcha-response.required' => 'Please complete the captcha verification.',
            'g-recaptcha-response.captcha' => 'Captcha verification failed. Please try again.'
        ]);

        // Queue a single email that will be sent to both admin and user
        Mail::queue(new ContactFormSubmission($validated));

        return redirect()->back()->with('success', 'Thank you for contacting us! We will get back to you soon.');
    }
}