<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Mail\ResetPassword;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Mail;
use Illuminate\View\View;
use Illuminate\Support\Facades\Validator;
use App\Models\User;

class PasswordResetLinkController extends Controller
{
    public function create(): View
    {
        return view('auth.forgot_password');
    }

    public function store(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'g-recaptcha-response' => 'required|captcha'
        ], [
            'email.required' => 'Please enter your email address.',
            'email.email' => 'Please enter a valid email address.',
            'g-recaptcha-response.required' => 'Please complete the captcha verification.',
            'g-recaptcha-response.captcha' => 'Captcha verification failed. Please try again.'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator);
        }

        // Find the user first
        $user = User::where('email', $request->email)->first();

        if ($user) {
            // Generate the token using the password broker
            $token = Password::createToken($user);
            
            // Queue the password reset email
            Mail::to($user)->queue(new ResetPassword($token, $user->email));
        }

        // Always return the same message whether the email exists or not
        return back()->with('status', 'If your email address exists in our database, you will receive a password reset link shortly.');
    }
}