<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Laravel\Socialite\Facades\Socialite;
use App\Models\User;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Cookie;

class LoginUserController extends Controller
{
    public function loginView()
    {
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'g-recaptcha-response' => 'required|captcha'
        ], [
            'email.required' => 'The email field is required.',
            'email.email' => 'Please enter a valid email address.',
            'password.required' => 'The password field is required.',
            'g-recaptcha-response.required' => 'The captcha field is required.'
        ]);

        $credentials = $request->only('email', 'password');
        $remember = $request->has('remember_me');

        if (Auth::attempt($credentials, $remember)) {
            $request->session()->regenerate();
            
            if (Auth::user()->email_verified_at === null) {
                Auth::logout();
                return redirect()->route('verification.notice');
            }

            if (!Auth::user()->is_active) {
                Auth::logout();
                return redirect()->route('login')
                    ->withErrors(['message' => 'Your account is currently inactive. Please contact support.']);
            }

            if (Auth::user()->isAdmin()) {
                return redirect()->route('admin.dashboard');
            }

            return redirect()->route('home')->with('success', 'Welcome back! Continue your shopping experience with us.');
        }

        return redirect()->back()
            ->withInput($request->only('email'))
            ->withErrors(['message' => 'Incorrect email or password. Please try again.']);
    }

    public function redirectToGoogle()
    {
        return Socialite::driver('google')->redirect();
    }
    public function handleGoogleCallback()
    {
        $googleUser = Socialite::driver('google')->user();
        $user = User::where('email', $googleUser->email)->first();

        if ($user) {
            if (!$user->is_active) {
                return redirect()->route('login')
                    ->withErrors(['message' => 'Your account is currently inactive. Please contact support.']);
            }

            Auth::login($user, true);

            if ($user->isAdmin()) {
                return redirect()->route('admin.dashboard');
            }

            return redirect()->route('home')->with('success', 'Welcome back! Continue your shopping experience with us.');
        } else {
            return redirect()->route('login')
                ->with('error', 'No account found for this Google account. Please register first.')
                ->with('google_email', $googleUser->email);
        }
    }

    public function logout(Request $request)
    {
        // Clear 2FA session and cookie
        Session::forget('2fa_verified');
        Cookie::queue(Cookie::forget('remember_2fa'));
        
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        return redirect()->route('login')->with('info', 'You have logged out!');
    }
}