<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\User;
use App\Models\Product;
use App\Models\ProductVariation;
use App\Models\OrderItem;
use App\Models\Blogs;
use App\Models\BlogComment;
use Illuminate\Support\Facades\DB;
 

class Generals extends Controller
{
    public function dashboard()
    {
        $today = now();
        $currentMonthStart = $today->copy()->startOfMonth();
        
        // Monthly metrics (current month only)
        $monthlyMetrics = [
            'total_orders' => Order::whereMonth('created_at', $today->month)
                ->whereYear('created_at', $today->year)
                ->count(),
            
            'total_revenue' => Order::whereMonth('created_at', $today->month)
                ->whereYear('created_at', $today->year)
                ->whereIn('order_status', ['processing', 'shipped', 'delivered'])
                ->sum('total'),
        ];

        // All-time metrics
        $allTimeMetrics = [
            'total_customers' => User::where('role', User::ROLE_USER)->count(),
            'total_products' => Product::count(),
        ];

        // Growth rates for today compared to yesterday
        $growthRates = [
            'orders_growth' => $this->calculateDailyGrowth('orders'),
            'revenue_growth' => $this->calculateDailyGrowth('revenue'),
            'customers_growth' => $this->calculateDailyGrowth('customers'),
            'products_growth' => $this->calculateDailyGrowth('products')
        ];

        // Combine metrics for view
        $metrics = [
            'monthly' => $monthlyMetrics,
            'all_time' => $allTimeMetrics,
            'growth' => $growthRates
        ];

        // Chart Data
        $charts = [
            'revenue' => $this->getRevenueChartData(),
            'order_status' => $this->getOrderStatusChartData(),
            'top_products' => $this->getTopProductsChartData(),
            'payment_methods' => $this->getPaymentMethodsChartData()
        ];

        // Get latest orders and total count
        $latest_orders = Order::with(['user'])
            ->latest()
            ->take(10)
            ->get();
        
        $total_orders = Order::count();

        // Get stock warnings
        $stock_warnings = ProductVariation::with(['product' => function($q) {
                // Eager load only necessary product fields
                $q->select('id', 'title');
            }])
            ->where('in_stock', '<=', ProductVariation::LOW_STOCK_THRESHOLD)
            ->orderBy('in_stock', 'asc') // Show lowest stock first
            ->get()
            ->map(function($variation) {
                $stockStatus = $variation->in_stock === 0 ? 'Out of Stock' : 
                    ($variation->in_stock <= ProductVariation::CRITICAL_STOCK_THRESHOLD ? 'Critical' : 'Low');
                
                $itemsText = $variation->in_stock === 0 ? 'out of stock' : 
                    ($variation->in_stock === 1 ? '1 item left' : $variation->in_stock . ' items left');

                return [
                    'product_id' => $variation->product_id,
                    'product' => $variation->product->title,
                    'variant' => $variation->unit_value . ' ' . $variation->unit_type,
                    'stock' => $variation->in_stock,
                    'stock_text' => $itemsText,
                    'level' => $variation->in_stock === 0 ? 'critical' : 
                        ($variation->in_stock <= ProductVariation::CRITICAL_STOCK_THRESHOLD ? 'critical' : 'low')
                ];
            });

        // Blog Statistics
        $blog_stats = [
            'total_posts' => Blogs::count(),
            'total_comments' => BlogComment::count(),
            'total_reactions' => DB::table('blog_reactions')->count()
        ];

        return view('admin.dashboard', compact(
            'metrics',
            'charts',
            'latest_orders',
            'total_orders',
            'stock_warnings',
            'blog_stats'
        ));
    }

    private function calculateDailyGrowth($metric)
    {
        $today = now();
        $yesterday = $today->copy()->subDay();
        
        // Only calculate growth if we're still in the same month
        if ($yesterday->month !== $today->month) {
            return 0; // First day of month, no growth to calculate
        }

        switch ($metric) {
            case 'orders':
                $today_value = Order::whereDate('created_at', $today)->count();
                $yesterday_value = Order::whereDate('created_at', $yesterday)->count();
                break;
            
            case 'revenue':
                $today_value = Order::whereDate('created_at', $today)
                    ->whereIn('order_status', ['processing', 'shipped', 'delivered'])
                    ->sum('total');
                $yesterday_value = Order::whereDate('created_at', $yesterday)
                    ->whereIn('order_status', ['processing', 'shipped', 'delivered'])
                    ->sum('total');
                break;
            
            case 'customers':
                $today_value = User::where('role', User::ROLE_USER)
                    ->whereDate('created_at', $today)->count();
                $yesterday_value = User::where('role', User::ROLE_USER)
                    ->whereDate('created_at', $yesterday)->count();
                break;
            
            case 'products':
                $today_value = Product::whereDate('created_at', $today)->count();
                $yesterday_value = Product::whereDate('created_at', $yesterday)->count();
                break;
            
            default:
                return 0;
        }

        // If no activity yesterday but some today, show max 100% growth
        if ($yesterday_value == 0) {
            return $today_value > 0 ? 100 : 0;
        }

        // Calculate growth percentage
        $growth = (($today_value - $yesterday_value) / $yesterday_value) * 100;
        
        // Cap growth at +/- 100%
        if ($growth > 100) return 100;
        if ($growth < -100) return -100;
        
        return round($growth, 1);
    }

    private function getRevenueChartData()
    {
        // Get all days in current month
        $currentMonth = now();
        $daysInMonth = $currentMonth->daysInMonth;
        $monthStart = $currentMonth->copy()->startOfMonth();
        
        // Generate all days of the month
        $days = collect(range(1, $daysInMonth))->map(function($day) use ($monthStart) {
            $date = $monthStart->copy()->addDays($day - 1);
            return [
                'label' => $date->format('M d'),
                'date' => $date->format('Y-m-d')
            ];
        });

        // Get daily revenue for current month
        $revenues = Order::whereMonth('created_at', $currentMonth->month)
            ->whereYear('created_at', $currentMonth->year)
            ->whereIn('order_status', ['processing', 'shipped', 'delivered'])
            ->selectRaw('DATE(created_at) as date, SUM(total) as total')
            ->groupBy('date')
            ->pluck('total', 'date');

        return [
            'labels' => $days->pluck('label'),
            'data' => $days->map(function($day) use ($revenues) {
                return $revenues[$day['date']] ?? 0;
            })
        ];
    }

    private function getOrderStatusChartData()
    {
        $today = now();
        $statuses = Order::whereMonth('created_at', $today->month)
            ->whereYear('created_at', $today->year)
            ->selectRaw('order_status, COUNT(*) as count')
            ->groupBy('order_status')
            ->pluck('count', 'order_status');

        return [
            'labels' => ['Pending', 'Processing', 'Shipped', 'Delivered', 'Cancelled'],
            'data' => [
                $statuses['pending'] ?? 0,
                $statuses['processing'] ?? 0,
                $statuses['shipped'] ?? 0,
                $statuses['delivered'] ?? 0,
                $statuses['cancelled'] ?? 0
            ]
        ];
    }

    private function getTopProductsChartData()
    {
        $today = now();
        $topProducts = OrderItem::with('product')
            ->whereHas('order', function($query) use ($today) {
                $query->whereMonth('created_at', $today->month)
                    ->whereYear('created_at', $today->year);
            })
            ->select('product_id', DB::raw('SUM(quantity) as total_sold'))
            ->groupBy('product_id')
            ->orderByDesc('total_sold')
            ->take(5)
            ->get();

        return [
            'labels' => $topProducts->pluck('product.title'),
            'data' => $topProducts->pluck('total_sold')
        ];
    }

    private function getPaymentMethodsChartData()
    {
        $today = now();
        
        // Monthly payment distribution based on orders table only
        $payments = Order::whereMonth('created_at', $today->month)
            ->whereYear('created_at', $today->year)
            ->selectRaw('payment_method, COUNT(*) as count')
            ->groupBy('payment_method')
            ->pluck('count', 'payment_method');

        $labels = ['sslcommerz' => 'SSLCommerz', 'cod' => 'Cash On Delivery'];
        $data = [
            $payments['sslcommerz'] ?? 0,
            $payments['cod'] ?? 0
        ];

        return [
            'labels' => array_values($labels),
            'data' => $data
        ];
    }

    
}